/*
 * logdev - Message logging driver for debug and timestamp
 * Copyright (C) 2012 Panasonic Corporation
 *
 * [Specification]
 *   - store written message into device file
 *   - store timestamp, time-diff and thread-id automatically (configurable)
 *   - minor device files are independent: log buffers are separated.
 *   - with ring-buffer mode, log area uses as ring-buffer, not become full.
 *
 * [How to use]
 *   write by command line:
 *     - echo "message" > /dev/log
 *   write by C program:
 *     - int fd = open("/dev/log", O_WRONLY);
 *     - write(fd, log_buffer, strlen(log_buffer));
 *     - (repeat writing...)
 *     - close(fd);
 *
 *   read by command line:
 *     - cat /dev/log
 *   read by C program:
 *     - int fd = open("/dev/log", O_RDONLY);
 *     - read(fd, log_buffer, sizeof(log_buffer));
 *     - (repeat reading until read() returns 0)
 *     - close(fd);
 *
 *   clean log buffer by command line:
 *     - echo 0 > /proc/logdev
 *       # where `0' means minor number.
 *   clean and change log buffer size by command line:
 *     - echo 0 1024 > /proc/logdev
 *       # where `0' means minor number, `1024' means buffer size.
 *
 *   show buffer information by command line:
 *     - cat /proc/logdev
 *
 *****************************************************************************/

#define __NO_VERSION__ /* Must be first in all but one driver source file! */

#include <linux/kernel.h>       /* for kernel space drivers */
#include <linux/syscalls.h>     /* for sys_gettid() */
#include <linux/module.h>       /* for EXPORT_SYMBOL() */
#include <linux/fs.h>           /* for file operaitons */
#include <linux/vmalloc.h>      /* for vmalloc()/vfree() */
#include <linux/slab.h>         /* for kmalloc()/kfree() */
#include <linux/proc_fs.h>      /* for create_proc_entry() */
#include <asm/uaccess.h>        /* for copy_from_user()/copy_to_user() */

#include <linux/iosc/iosc-devices.h> /* for LOG_DEV_MAJOR macro */

#define NUM_MINOR             8
#define LOG_DEFAULT_SIZE      (1*1024*1024)
#define LOG_BOL_MARK          0xfefefefe

#define ERR(fmt, args...) printk(KERN_ERR "[logdev] " fmt "\n", ##args)

/* semaphore to open/write exclusively */
static struct semaphore logdev_sem[NUM_MINOR];

/* buffer information for each minor */
struct logbuf {
  int log_size;
  char *top;
  char *bottom;
  char *write;
  struct timeval prev_time;
  int timestamp_flag;
  int timediff_flag;
  int tid_flag;
  int is_looped;                /* flag if write looped whole buffer */
};
static struct logbuf logbuf[NUM_MINOR];

/* read pointer information for each open(RD_ONLY) */
struct logdev_priv_struct {
  char *last_read_ptr;
  int wrapped;
  int minor;
};


/* module related and driver method functions */
static int logdev_init( void );
static ssize_t logdev_write(struct file *file, const char __user *buf, size_t count, loff_t *offset);
static ssize_t logdev_read(struct file *file, char __user *buf, size_t count, loff_t *offset);

/*
 * alloc log buffer and set buffer addresses in global var logbuf[minor].
 */
static int logdev_init_buf(int minor)
{
  struct logbuf *logpt;

  if (minor < 0 || NUM_MINOR <= minor) { /* out of minor range */
    return -ENODEV;
  }

  logpt = &logbuf[minor];

  if (!logpt->top) {          /* first access, do initialize */
    char *buf = vmalloc(logbuf[minor].log_size);
    if (!buf) {
      ERR("cannot allocate memory for logging buffer");
      return -ENOMEM;
    }
    logpt->top = buf;
    logpt->bottom = buf + logpt->log_size;
    logpt->write = buf;
    logpt->prev_time.tv_sec = 0;
    logpt->prev_time.tv_usec = 0;
    logpt->is_looped = 0;
  }

  return 0;
}

/*****************************************************************
 * Method functions for Linux-side device
 *****************************************************************/

/*
 * [Method] open
 * RETURN: 0 if success
 *         under 0 if error occured
 */
static int
logdev_open(struct inode *inode, struct file *file)
{
  int minor = MINOR(inode->i_rdev);
  struct logdev_priv_struct *priv;
  struct logbuf *logpt;

  if (minor < 0 || NUM_MINOR <= minor) {
    return -ENODEV;
  }

  logpt = &logbuf[minor];

  priv = kmalloc(sizeof(struct logdev_priv_struct), GFP_KERNEL);
  if (!priv) {
      ERR("cannot allocate memory for private data");
      return -ENOMEM;
  }
  priv->minor = MINOR(inode->i_rdev);
  priv->last_read_ptr = 0;
  file->private_data = (void *)priv;

  return 0;

}

/*
 * [Method] close
 * RETURN: length if success to log
 *         under 0 if error occured
 */
static int
logdev_release(struct inode *inode, struct file *file)
{
  struct logdev_priv_struct *priv = (struct logdev_priv_struct *)(file->private_data);

  if (priv) {
    kfree(priv);
    priv = NULL;
  }

  /* do not free logbuf[minor] not to lost written logs */
  return 0;
}

/*
 * [Method] write
 * RETURN: length if success to log
 *         under 0 if error occured
 */

/* sub routine: calculate diff between timeval */
static void calc_time_diff(struct timeval *prev, struct timeval *now, struct timeval *diff)
{
  int sec, usec;

  if (prev->tv_sec == 0 && prev->tv_usec== 0) {
    sec = usec = 0;
  } else {
    sec = now->tv_sec - prev->tv_sec;
    usec = now->tv_usec - prev->tv_usec;
    if (usec < 0) {
      sec--;
      usec += 1000*1000;
    }
  }

  diff->tv_sec = sec;
  diff->tv_usec = usec;
}

static char *write_to_ringbuffer(char *top, char *bottom, char *write,
                                 char *buf, int count, int *looped)
{
  int logsize = bottom-top;
  int remain;

  while (count > logsize) {
    count -= logsize;
    *looped = 1;
  }

  remain = bottom-write;
  if (remain < count) {
    memcpy(write, buf, remain);
    buf += remain;
    count -= remain;
    write = top;
    *looped = 1;
  }

  memcpy(write, buf, count);
  write += count;

  return write;
}

static ssize_t write_inner(int minor, char *buf, size_t count)
{
  int ret;
  int header_len = 0;
  int tid;
  struct timeval now;
  char *write_ptr;
  char header[100];
  struct logbuf *logpt = &logbuf[minor];

  /* get informations before entering critical section */
  tid = sys_gettid();
  do_gettimeofday(&now);

  down(&logdev_sem[minor]);     /* ++++++++++++++++ */

  ret = logdev_init_buf(minor);
  if (ret) {
    count = ret; goto EXIT;
  }

  /* store timestamp */
  if (logpt->timestamp_flag || logpt->timediff_flag) {
    header_len += snprintf(header, sizeof(header)-header_len-1,
                          "%d.%06d\t", (int)now.tv_sec, (int)now.tv_usec);

    /* store timestamp-diff */
    if (logpt->timediff_flag) {
      struct timeval diff;
      calc_time_diff(&logpt->prev_time, &now, &diff);
      header_len += snprintf(header+header_len, sizeof(header)-header_len-1,
                             "%d.%06d\t", (int)diff.tv_sec, (int)diff.tv_usec);
      logpt->prev_time = now;
    }
  }

  /* store thread-id */
  if (logpt->tid_flag) {
    header_len += snprintf(header+header_len, sizeof(header)-header_len-1,
                           "#%d\t", tid);
  }

  write_ptr = logpt->write;
  write_ptr = write_to_ringbuffer(logpt->top, logpt->bottom, write_ptr,
                                  header, header_len, &logpt->is_looped);
  write_ptr = write_to_ringbuffer(logpt->top, logpt->bottom, write_ptr,
                                  buf, count, &logpt->is_looped);

  logpt->write = write_ptr;

 EXIT:
  up(&logdev_sem[minor]);       /* ---------------- */
  return count;
}

static ssize_t
logdev_write(struct file *file, const char __user *buf, size_t count, loff_t *offset)
{
  struct logdev_priv_struct *priv = (struct logdev_priv_struct *)(file->private_data);
  int minor = priv->minor;
  char body[count+2];

  /* copy log body */
  if (copy_from_user(body, buf, count)) {
    ERR("cannot copy log-body from user-space");
    return -EFAULT;
  }
  if (body[count-1] != '\n') {  /* add newline kindly (?) */
    body[count] = '\n';
    count++;
  }

  return write_inner(minor, body, count);
}

/*
 * [Method] read
 * RETURN: length from one entry of log
 *         under 0 if error occured
 */

static char overtaken_message[] = "...(overtaken)...\n";

static ssize_t
logdev_read(struct file *file, char __user *buf, size_t count, loff_t *offset)
{
  int ret;
  int read_len = 0;

  struct logdev_priv_struct *priv = (struct logdev_priv_struct *)(file->private_data);
  char *read_ptr = priv->last_read_ptr;
  int minor = priv->minor;
  struct logbuf *logpt = &logbuf[minor];

  down(&logdev_sem[minor]);       /* ++++++++++++++++ */

  if (!logpt->top || read_ptr == logpt->write) { /* no log, or already read whole logs */
    ret = 0; goto EXIT;
  }

  /* initialize read_ptr if reading first time with this file descriptor */
  if (read_ptr == 0) {
    if (logpt->is_looped) {
      read_ptr = logpt->write + 1;
      priv->wrapped = 1;
    } else {
      read_ptr = logpt->top;
      priv->wrapped = 0;
    }
  }

  if (priv->wrapped && read_ptr < logpt->write) { /* overtaken!! */
    if (copy_to_user(buf, overtaken_message, sizeof(overtaken_message)+1)) {
      ERR("failed to overtaken message");
      ret = -EFAULT; goto EXIT;
    }
    read_ptr = logpt->write;
    ret = sizeof(overtaken_message)+1; goto EXIT;
  }

  /* if read_ptr is higher than write_ptr */
  if (priv->wrapped) {
    int remain = logpt->bottom - read_ptr;
    if (count <= remain) {      /* normal copy */
      if (copy_to_user(buf, read_ptr, count)) {
        ERR("failed to copy log");
        ret = -EFAULT; goto EXIT;
      }
      read_len += count;
      read_ptr += count;
    } else {                    /* split copy */
      if (copy_to_user(buf, read_ptr, remain)) {
        ERR("failed to copy log");
        ret = -EFAULT; goto EXIT;
      }
      count -= remain;
      read_len += remain;
      priv->wrapped = 0;
      read_ptr = logpt->top;
    }
  }

  /* if read_ptr is lower than write_ptr */
  if (!priv->wrapped) {
    int copy_len = count;
    int written = logpt->write - read_ptr;
    copy_len = count < written ? count: written;
    if (copy_to_user(buf+read_len, read_ptr, copy_len)) {
      ERR("failed to copy log");
      ret = -EFAULT; goto EXIT;
    }
    read_len += copy_len;
    read_ptr += copy_len;
  }

  ret = read_len;

 EXIT:
  priv->last_read_ptr = read_ptr;
  up(&logdev_sem[minor]);       /* ---------------- */
  return ret;
}

/*
 * [Method] seek
 */
static loff_t
logdev_seek(struct file *file, loff_t off, int whence)
{
  struct logdev_priv_struct *priv = (struct logdev_priv_struct *)(file->private_data);
  int minor = priv->minor;
  struct logbuf *logpt = &logbuf[minor];

  down(&logdev_sem[minor]);       /* ++++++++++++++++ */

  if (logpt->top) {
    /* ok, log buffre already allocated */
    switch (whence) {
    case SEEK_END:
      priv->last_read_ptr = logpt->write;
      priv->wrapped = 0;
      break;
    default:                  /* do nothing, only care about SEEK_END */
      break;
    }
  }

  up(&logdev_sem[minor]);       /* ---------------- */

  return priv->last_read_ptr - logpt->top;
}


/*
 * [proc] get logbuf information (size, alloc_status and so on)
 */
#define YESNO(bool) (bool ? "Y" : "n")

static int
logdev_proc_read(char *page, char **start, off_t off, int count, int *eof, void *data)
{
  char *p = page;
  int i, len;

  p += sprintf(p, "minor\tsize\talloc?\ttop\t\tbottom\t\twrite\t\tlooped?\n");
  for (i=0; i<NUM_MINOR; i++) {
    struct logbuf *logpt = &logbuf[i];

    p += sprintf(p, "%d\t%d\t%s\t0x%08x\t0x%08x\t0x%08x\t%s\n",
                 i, logpt->log_size, YESNO(logpt->top),
                 (unsigned int)logpt->top, (unsigned int)logpt->bottom,
                 (unsigned int)logpt->write,
                 YESNO(logpt->is_looped));
  }

  len = (p - page) - off;
  if (len < 0)
    len = 0;

  *eof = (len <= count) ? 1 : 0;
  *start = page + off;

  return len;
}

/*
 * [proc] change clear log buffer, or change log buffer size if specified
 */
static int logdev_proc_write(struct file *file, const char __user *buffer, unsigned long count, void *data)
{
  int len = count;
  int minor = -1;
  int log_size = 0;
  char buf[100];
  struct logbuf *logpt;

  if (len <= 0) {
    return len;
  }

  if (len > sizeof(buf)-1) {
    len = sizeof(buf)-1;
  }
  if (copy_from_user(buf, buffer, len)) {
    return -EFAULT;
  }
  buf[len] = 0;

  sscanf(buf, "%d %d", &minor, &log_size);
  if (minor < 0 || NUM_MINOR <= minor) {
    return -ENODEV;
  }

  down(&logdev_sem[minor]);   /* ++++++++++++++++ */

  logpt = &logbuf[minor];

  if (log_size <= 0) {          /* only clean log buffer */
    if (logpt->top) {
      logpt->write = logpt->top;
      logpt->is_looped = 0;
    }
  } else {                      /* change log buffer size and clean */
    if (logpt->top) {    /* free if buffer already allocated */
      vfree(logpt->top);
      logpt->top = NULL;
    }
    logpt->log_size = log_size;
  }

  up(&logdev_sem[minor]);   /* ---------------- */

  return count;
}




/* file operation structure */
static struct file_operations logdev_fops = {
  open:    logdev_open,
  release: logdev_release,
  write:   logdev_write,
  read:    logdev_read,
  llseek:  logdev_seek,

};

/*
 * driver initializer
 */
static int logdev_init( void )
{
  int i;
  int result;
  struct proc_dir_entry *proc;

  result = register_chrdev(LOG_DEV_MAJOR, "logdev", &logdev_fops);
  if (result < 0) {
    ERR("failed to register device\n");
    return -1;
  }

  proc = create_proc_entry("logdev", S_IFREG | S_IALLUGO, NULL);
  if (!proc) {
    result = -ENOMEM;
    goto EXIT;
  }
  proc->read_proc = logdev_proc_read;
  proc->write_proc = logdev_proc_write;

  for (i=0; i<NUM_MINOR; i++) {
    sema_init(&logdev_sem[i], 1);
    logbuf[i].log_size = LOG_DEFAULT_SIZE;
    logbuf[i].timestamp_flag = 1;
    logbuf[i].timediff_flag = 1;
    logbuf[i].tid_flag = 1;
  }

  return 0;

 EXIT:
  unregister_chrdev(LOG_DEV_MAJOR, "logdev");
  return result;
}

module_init(logdev_init);

#include <stdarg.h>

/*
 * global library
 *-----------------------------------------------------------------------------
 * function: 
 * argument: nothing
 * return  :  0         : success
 *            under 0   : fail
 * comment :
 */
void logdev(int minor, const char *fmt, ...)
{
  char buf[256];
  int size;
  va_list args;

  if (minor < 0 || NUM_MINOR <= minor)
    return;

  if (logdev_init_buf(minor))
    return;

  va_start(args, fmt);
  size = vsnprintf(buf, sizeof(buf)-1, fmt, args);
  write_inner(minor, buf, size);

  va_end(args);
}

EXPORT_SYMBOL(logdev);
